/**
 * @file
 * Javascript for the Google map formatter.
 */

(function ($, Drupal) {

  'use strict';

  /**
   * @namespace
   */
  Drupal.geolocation = Drupal.geolocation || {};

  /**
   * Attach Google Maps formatter functionality.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attaches Google Maps formatter functionality to relevant elements.
   */
  Drupal.behaviors.geolocationGoogleMaps = {
    attach: function (context, drupalSettings) {
      if (typeof Drupal.geolocation.loadGoogle === 'function') {
        // First load the library from google.
        Drupal.geolocation.loadGoogle(function () {
          initialize(drupalSettings.geolocation.maps, context);
        });
      }
    }
  };

  /**
   * Runs after the Google Maps API is available
   *
   * @param {GeolocationMap[]} mapSettings - The geolocation map objects.
   * @param {object} context - The html context.
   */
  function initialize(mapSettings, context) {

    /* global google */

    $.each(
      mapSettings,

      /**
       * @param {string} mapId - Current map ID
       * @param {GeolocationMap} map - Single map settings Object
       */
      function (mapId, map) {
        map.id = mapId;

        // Get the map container.
        /** @type {jQuery} */
        var mapWrapper = $('#' + mapId, context).first();

        if (
          mapWrapper.length
          && !mapWrapper.hasClass('geolocation-processed')
        ) {
          if (
            mapWrapper.data('map-lat')
            && mapWrapper.data('map-lng')
          ) {
            map.lat = Number(mapWrapper.data('map-lat'));
            map.lng = Number(mapWrapper.data('map-lng'));
          }
          else {
            map.lat = 0;
            map.lng = 0;
          }

          map.container = mapWrapper.children('.geolocation-google-map');

          // Add the map by ID with settings.
          map.googleMap = Drupal.geolocation.addMap(map);

          if (mapWrapper.find('.geolocation-common-map-locations .geolocation').length) {

            /**
             * Result handling.
             */
              // A Google Maps API tool to re-center the map on its content.
            var bounds = new google.maps.LatLngBounds();

            Drupal.geolocation.removeMapMarker(map);

            /*
             * Add the locations to the map.
             */
            mapWrapper.find('.geolocation-common-map-locations .geolocation').each(function (index, location) {

              /** @type {jQuery} */
              location = $(location);
              var position = new google.maps.LatLng(Number(location.data('lat')), Number(location.data('lng')));

              bounds.extend(position);

              if (location.data('set-marker') === 'false') {
                return;
              }

              /** @type {GoogleMarkerSettings} */
              var markerConfig = {
                position: position,
                map: map.googleMap,
                title: location.children('.location-title').text(),
                infoWindowContent: location.html(),
                infoWindowSolitary: true
              };

              var skipInfoWindow = false;
              if (location.children('.location-content').text().trim().length < 1) {
                skipInfoWindow = true;
              }

              Drupal.geolocation.setMapMarker(map, markerConfig, skipInfoWindow);
            });

            map.googleMap.fitBounds(bounds);

            if (map.settings.google_map_settings.zoom) {
              google.maps.event.addListenerOnce(map.googleMap, 'zoom_changed', function () {
                if (map.settings.google_map_settings.zoom < map.googleMap.getZoom()) {
                  map.googleMap.setZoom(map.settings.google_map_settings.zoom);
                }
              });
            }
          }

          // Set the already processed flag.
          map.container.addClass('geolocation-processed');
        }
      }
    );
  }
})(jQuery, Drupal);
