<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  Captcha.recaptcha
 *
 * @copyright   (C) 2011 Open Source Matters, Inc. <https://www.joomla.org>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Os_lib\Plugin\Captcha\ReCaptchaOrdasoft;

use Joomla\CMS\Application\CMSWebApplicationInterface;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Event\DispatcherInterface;
use Joomla\Utilities\IpHelper;
use ReCaptcha\ReCaptcha as Captcha2;
use ReCaptcha\RequestMethod;

use Joomla\Application\SessionAwareWebApplicationInterface;
use Joomla\CMS\Application\CMSApplicationInterface;
use Joomla\CMS\Captcha\Captcha;
use Joomla\CMS\Captcha\CaptchaRegistry;
use Joomla\CMS\Extension\PluginInterface;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\Field\CaptchaField;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Http\HttpFactory;
use Joomla\Registry\Registry;
use Joomla\Utilities\ArrayHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Recaptcha Plugin
 * Based on the official recaptcha library( https://packagist.org/packages/google/recaptcha )
 *
 * @since  2.5
 */
final class ReCaptchaOrdasoft extends CMSPlugin
{
    /**
     * Load the language file on instantiation.
     *
     * @var    boolean
     * @since  3.1
     */
    protected $autoloadLanguage = true;

    /**
     * The http request method
     *
     * @var    RequestMethod
     * @since  4.3.0
     */
    private $requestMethod;
    
    /**
    * Remote service error codes
    *
    * @var	string[]
    * @since  1.0.0
    */
    private const ERROR_CODES = [
            'missing-input-secret',
            'invalid-input-secret',
            'missing-input-response',
            'invalid-input-response',
            'bad-request',
            'timeout-or-duplicate',
    ];

    /**
     * Hash of the script file.
     *
     * @var	 string
     * @since  1.0.0
     */
    private const SCRIPT_HASH = '69134694';

    /**
     * Application instance.
     *
     * @var	 CMSApplicationInterface
     * @since  1.0.0
     */
    private $app;

    /**
     * Plugin parameters.
     *
     * @var	 Registry
     * @since  1.0.0
     */
    public $params;

    /**
     * HTTP factory instance.
     *
     * @var	 HttpFactory
     * @since  1.0.0
     */
    private $httpFactory;

    /**
     * Alternative Captcha instance, if set.
     *
     * @var	 ?Captcha
     * @since  1.0.0
     */
    private $captcha;

    /**
     * Constructor.
     *
     * @param   DispatcherInterface  $dispatcher     The dispatcher
     * @param   array                $config         An optional associative array of configuration settings
     * @param   RequestMethod        $requestMethod  The http request method
     *
     * @since   4.3.0
     */
    public function __construct(DispatcherInterface $dispatcher, array $config, RequestMethod $requestMethod, CMSApplicationInterface $app, Registry $params, HttpFactory $httpFactory)
    {
        parent::__construct($dispatcher, $config);

        $this->requestMethod = $requestMethod;
        
        $this->app = $app;
//        $this->params = $params;
        $this->httpFactory = $httpFactory;
    }

    /**
     * Reports the privacy related capabilities for this plugin to site administrators.
     *
     * @return  array
     *
     * @since   3.9.0
     */
    public function onPrivacyCollectAdminCapabilities()
    {
        return [
            $this->getApplication()->getLanguage()->_('PLG_CAPTCHA_RECAPTCHA') => [
                $this->getApplication()->getLanguage()->_('PLG_RECAPTCHA_PRIVACY_CAPABILITY_IP_ADDRESS'),
            ],
        ];
    }

    /**
     * Initializes the captcha
     *
     * @param   string  $id  The id of the field.
     *
     * @return  Boolean True on success, false otherwise
     *
     * @since   2.5
     * @throws  \RuntimeException
     */
    public function onInit($id = 'dynamic_recaptcha_1')
    {
        $version    = $this->params->get('version', '2.0');
        if($version == '2.0'){
//            var_dump(!$this->app instanceof CMSWebApplicationInterface); exit;
//            $app = $this->app();
            $app = $this->app;

            if (!$app instanceof CMSWebApplicationInterface) {
                return false;
            }
//            var_dump($this->params); exit;
            $pubkey = $this->params->get('public_key', '');

            if ($pubkey === '') {
                throw new \RuntimeException($app->getLanguage()->_('PLG_RECAPTCHA_ERROR_NO_PUBLIC_KEY'));
            }


            
            $apiSrc = 'https://www.google.com/recaptcha/api.js?onload=JoomlainitReCaptcha2&render=explicit&hl='
                    . $app->getLanguage()->getTag();
            

            // Load assets, the callback should be first
            $app->getDocument()->getWebAssetManager()
                ->registerAndUseScript('plg_captcha_recaptcha', 'plg_captcha_recaptcha_ordasoft/recaptchaV2.js', [], ['defer' => true])
                ->registerAndUseScript('plg_captcha_recaptcha.api', $apiSrc, [], ['defer' => true], ['plg_captcha_recaptcha']);

            return true;
        }elseif($version == '3.0'){
            if ($this->shouldShowCaptcha())
		{
			return $this->getCaptcha()->initialise($id);
		}

		if (!$siteKey = $this->params->get('public_key'))
		{
			return false;
		}

		if (!$this->app instanceof CMSWebApplicationInterface)
		{
			return false;
		}

		$document = $this->app->getDocument();


		$document->addScriptOptions('plg_captcha_recaptcha_ordasoft.siteKey', $siteKey);
		$document->addScriptOptions('plg_captcha_recaptcha_ordasoft.triggerMethod', $this->params->get('triggerMethod', 'submit'));
		$assetManager = $document->getWebAssetManager();

		if (!$assetManager->assetExists('script', 'plg_captcha_recaptcha_ordasoft.api.js'))
		{
			$languageTag = $this->app->getLanguage()->getTag();
			$assetManager->registerAsset(
				'script',
				'plg_captcha_recaptcha_ordasoft.api.js',
				'https://www.google.com/recaptcha/api.js?hl=' . $languageTag . '&render=' . $siteKey,
				[],
				['defer' => true, 'referrerpolicy' => 'no-referrer'],
				['core']
			);
		}

		if (!$assetManager->assetExists('script', 'plg_captcha_recaptcha_ordasoft.recaptchaV3.js'))
		{
			$assetManager->registerAsset(
				'script',
				'plg_captcha_recaptcha_ordasoft.recaptchaV3.js',
				'plg_captcha_recaptcha_ordasoft/recaptchaV3.js',
				['version' => self::SCRIPT_HASH],
				['type' => 'module'],
				['plg_captcha_recaptcha_ordasoft.api.js', 'core']
			);
		}

		$assetManager->useAsset('script', 'plg_captcha_recaptcha_ordasoft.api.js');
		$assetManager->useAsset('script', 'plg_captcha_recaptcha_ordasoft.recaptchaV3.js');

		return true;
        }
    }

    /**
     * Gets the challenge HTML
     *
     * @param   string  $name   The name of the field. Not Used.
     * @param   string  $id     The id of the field.
     * @param   string  $class  The class of the field.
     *
     * @return  string  The HTML to be embedded in the form.
     *
     * @since  2.5
     */
    public function onDisplay($name = null, $id = 'dynamic_recaptcha_1', $class = '')
    {
        
        $version    = $this->params->get('version', '2.0');
        if($version == '2.0'){
            $dom = new \DOMDocument('1.0', 'UTF-8');
            $ele = $dom->createElement('div');
            $ele->setAttribute('id', $id);

            
            $ele->setAttribute('class', ((trim($class) == '') ? 'g-recaptcha' : ($class . ' g-recaptcha')));
            $ele->setAttribute('data-sitekey', $this->params->get('public_key', ''));
            $ele->setAttribute('data-theme', $this->params->get('theme2', 'light'));
            $ele->setAttribute('data-size', $this->params->get('size', 'normal'));
            $ele->setAttribute('data-tabindex', $this->params->get('tabindex', '0'));
            $ele->setAttribute('data-callback', $this->params->get('callback', ''));
            $ele->setAttribute('data-expired-callback', $this->params->get('expired_callback', ''));
            $ele->setAttribute('data-error-callback', $this->params->get('error_callback', ''));
            

            $dom->appendChild($ele);

            return $dom->saveHTML($ele);
        }elseif($version == '3.0'){
            
            if ($this->shouldShowCaptcha())
            {
                    return $this->getCaptcha()->display($name, $id, $class);
            }

//            $this->loadLanguage();

            if (!$this->params->get('public_key'))
            {
                    return $this->render('nokey');
            }

            $attributes = [
                    'type' => 'hidden',
                    'class' => 'plg-captcha-recaptcha-ordasoft-hidden',
            ];

            if ($name !== null && $name !== '')
            {
                    $attributes['name'] = $name;
            }

            if ($id !== null && $id !== '')
            {
                    $attributes['id'] = $id;
            }

            $attributes = array_map([$this, 'escape'], $attributes);

            $html = '<input ' . ArrayHelper::toString($attributes) . '>';
            $html .= '<input type="hidden" name="plg_captcha_recaptcha_ordasoft_action" class="plg-captcha-recaptcha-ordasoft-action">';
            $html .= $this->render('noscript');


            return $html;
        }
    }
    
    /**
    * Alters form field.
    *
    * @param   CaptchaField       $field    Captcha field instance
    * @param   \SimpleXMLElement  $element  XML form definition
    *
    * @return  void
    *
    * @since  1.0.0
    */
   public function onSetupField(CaptchaField $field, \SimpleXMLElement $element)
   {
           if ($this->shouldShowCaptcha())
           {
                   $this->getCaptcha()->setupField($field, $element);

                   return;
           }

           $element['hiddenLabel'] = 'true';
   }

    /**
     * Calls an HTTP POST function to verify if the user's guess was correct
     *
     * @param   string  $code  Answer provided by user. Not needed for the Recaptcha implementation
     *
     * @return  True if the answer is correct, false otherwise
     *
     * @since   2.5
     * @throws  \RuntimeException
     */
    public function onCheckAnswer($code = null)
    {
        $input      = $this->getApplication()->getInput();
        $privatekey = $this->params->get('private_key');
        $version    = $this->params->get('version', '2.0');
        $remoteip   = IpHelper::getIp();
        $response   = null;
        $spam       = false;
        
        if($version == '2.0'){
            switch ($version) {
                case '2.0':
                    $response  = $code ?: $input->get('g-recaptcha-response', '', 'string');
                    $spam      = ($response === '');
                    break;
            }

            // Check for Private Key
            if (empty($privatekey)) {
                throw new \RuntimeException($this->getApplication()->getLanguage()->_('PLG_RECAPTCHA_ERROR_NO_PRIVATE_KEY'), 500);
            }

            // Check for IP
            if (empty($remoteip)) {
                throw new \RuntimeException($this->getApplication()->getLanguage()->_('PLG_RECAPTCHA_ERROR_NO_IP'), 500);
            }

            // Discard spam submissions
            if ($spam) {
                throw new \RuntimeException($this->getApplication()->getLanguage()->_('PLG_RECAPTCHA_ERROR_EMPTY_SOLUTION'), 500);
            }

            return $this->getResponse($privatekey, $remoteip, $response);
        }if($version == '3.0'){
            
            if ($this->shouldShowCaptcha())
            {
                    if ($answer = $this->getCaptcha()->checkAnswer($code))
                    {
                            $this->setShouldShowCaptcha(false);
                    }

                    return $answer;
            }

            $language = $this->app->getLanguage();
//            $this->loadLanguage();

            if ($code === null || $code === '')
            {
                    // No answer provided, form was manipulated.
                    throw new \RuntimeException($language->_('PLG_CAPTCHA_RECAPTCHA_ORDASOFT_ERROR_EMPTY_RESPONSE'));
            }

            if (!$this->params->get('private_key'))
            {
                    throw new \RuntimeException($language->_('PLG_CAPTCHA_RECAPTCHA_ORDASOFT_NO_SECRET_KEY'));
            }

            try
            {
                    $http = $this->httpFactory->getHttp();
            }
            catch (\RuntimeException $exception)
            {
                    if (\JDEBUG)
                    {
                            throw $exception;
                    }

                    // No HTTP transports supported.
                    return !$this->params->get('strictMode');
            }

            $data = [
                    'response' => $code,
                    'secret' => $this->params->get('private_key'),
            ];

            try
            {
                    $response = $http->post('https://www.google.com/recaptcha/api/siteverify', $data);
                    $body = json_decode($response->body);
            }
            catch (\RuntimeException $exception)
            {
                    if (\JDEBUG)
                    {
                            throw $exception;
                    }

                    // Connection or transport error.
                    return !$this->params->get('strictMode');
            }

            // Remote service error.
            if ($body === null)
            {
                    if (\JDEBUG)
                    {
                            throw new \RuntimeException($language->_('PLG_CAPTCHA_RECAPTCHA_ORDASOFT_ERROR_INVALID_RESPONSE'));
                    }

                    return !$this->params->get('strictMode');
            }

            if (!isset($body->success) || $body->success !== true)
            {
                    // If error codes are pvovided, use them for language strings.
                    if (!empty($body->{'error-codes'}) && \is_array($body->{'error-codes'}))
                    {
                            if ($errors = array_intersect($body->{'error-codes'}, self::ERROR_CODES))
                            {
                                    $error = $errors[array_key_first($errors)];

                                    throw new \RuntimeException($language->_('PLG_CAPTCHA_RECAPTCHA_ORDASOFT_ERROR_' . strtoupper(str_replace('-', '_', $error))));
                            }
                    }

                    return false;
            }

            $score = $this->params->get('score', 0.5);

            if (!\is_float($score) || $score < 0 || $score > 1)
            {
                    $score = 0.5;
            }

//            var_dump($body);
//            var_dump($this->app->getInput()->get('plg_captcha_recaptcha_os_action', '', 'RAW'));
//            exit;
//            if (!isset($body->action) || $body->action !== $this->app->getInput()->get('plg_captcha_recaptcha_os_action', '', 'RAW'))
//            {
//                    throw new \RuntimeException('PLG_CAPTCHA_RECAPTCHA_OS_ERROR_INVALID_ACTION');
//            }

            if (!isset($body->score) || $body->score < $score)
            {
                    if ($this->hasCaptcha())
                    {
                            $this->setShouldShowCaptcha(true);
                    }

                    throw new \RuntimeException($language->_('PLG_CAPTCHA_RECAPTCHA_ORDASOFT_ERROR_CAPTCHA_VERIFICATION'));
            }

            if ($this->hasCaptcha())
            {
                    $this->setShouldShowCaptcha(false);
            }

            return true;
        }
    }

    /**
     * Get the reCaptcha response.
     *
     * @param   string  $privatekey  The private key for authentication.
     * @param   string  $remoteip    The remote IP of the visitor.
     * @param   string  $response    The response received from Google.
     *
     * @return  bool True if response is good | False if response is bad.
     *
     * @since   3.4
     * @throws  \RuntimeException
     */
    private function getResponse(string $privatekey, string $remoteip, string $response)
    {
        $version = $this->params->get('version', '2.0');

        switch ($version) {
            case '2.0':
                $apiResponse = (new Captcha2($privatekey, $this->requestMethod))->verify($response, $remoteip);

                if (!$apiResponse->isSuccess()) {
                    foreach ($apiResponse->getErrorCodes() as $error) {
                        throw new \RuntimeException($error, 403);
                    }

                    return false;
                }

                break;
        }

        return true;
    }
    
    private function escape(?string $string)
    {
            return $string ? htmlspecialchars($string, \ENT_QUOTES|\ENT_SUBSTITUTE|\ENT_HTML5, 'UTF-8') : (string) $string;
    }

    private function render(string $layout)
    {
            $html = '';
            $file = PluginHelper::getLayoutPath('captcha', 'recaptcha_ordasoft', $layout);

            if (!is_file($file))
            {
                    return '';
            }

            $data = [
                    'language' => $this->app->getLanguage(),
            ];

            ob_start();

            (static function ()
            {
                    extract(func_get_arg(1));

                    include func_get_arg(0);
            })($file, $data);

            $html .= ob_get_clean();

            return $html;
    }

//    private function loadLanguage()
//    {
//            $this->app->getLanguage()->load('plg_captcha_recaptcha_os', \JPATH_ADMINISTRATOR);
//    }

    private function setShouldShowCaptcha(bool $value)
    {
            if (!$this->app instanceof SessionAwareWebApplicationInterface)
            {
                    return;
            }

            if ($value)
            {
                    $this->app->getSession()->set('plg_captcha_recaptcha_ordasoft.showCaptcha', true);

                    return;
            }

            $this->app->getSession()->remove('plg_captcha_recaptcha_ordasoft.showCaptcha');
    }

    private function shouldShowCaptcha()
    {
            if (!$this->hasCaptcha())
            {
                    return false;
            }

            if (!$this->app instanceof SessionAwareWebApplicationInterface)
            {
                    return false;
            }

            return $this->app->getSession()->has('plg_captcha_recaptcha_ordasoft.showCaptcha');
    }

    private function hasCaptcha()
    {
            if (!$captcha = $this->params->get('captcha'))
            {
                    return false;
            }

            if ($captcha === 'recaptcha_ordasoft')
            {
                    return false;
            }

            if (version_compare(\JVERSION, '5.0', '>='))
            {
                    $container = Factory::getContainer();

                    if ($container->has(CaptchaRegistry::class) && $container->get(CaptchaRegistry::class)->has($captcha))
                    {
                            return true;
                    }
            }

            return PluginHelper::isEnabled('captcha', $captcha);
    }

    private function getCaptcha()
    {
            if ($this->captcha === null)
            {
                    $this->captcha = Captcha::getInstance($this->params->get('captcha'));
            }

            return $this->captcha;
    }
}
